// 全局变量保存题库
let QA_DATA = [];
let qaBall = null;
let qaPopup = null;
let hideBallTimeout = null;

// -------------------- 1. 加载本地题库 questions.json --------------------
async function loadQuestions() {
  if (QA_DATA.length > 0) return;

  try {
    const url = chrome.runtime.getURL("questions.json");
    const res = await fetch(url);
    if (!res.ok) {
      console.error(
        "[划词题库小球] 加载题库失败，HTTP 状态码：",
        res.status
      );
      QA_DATA = [];
      return;
    }
    const data = await res.json();
    if (!Array.isArray(data)) {
      console.error("[划词题库小球] 题库格式不是数组，请检查 questions.json");
      QA_DATA = [];
      return;
    }
    QA_DATA = data;
    console.log("[划词题库小球] 题库已加载，数量：", QA_DATA.length);
  } catch (e) {
    console.error("[划词题库小球] 加载题库时出错：", e);
    QA_DATA = [];
  }
}

// -------------------- 2. 创建小球 DOM --------------------
function createBall() {
  if (qaBall) return qaBall;
  qaBall = document.createElement("div");
  qaBall.className = "qa-hover-ball";
  qaBall.style.display = "none";

  // 鼠标移入小球时，显示弹框
  qaBall.addEventListener("mouseenter", () => {
    if (hideBallTimeout) {
      clearTimeout(hideBallTimeout);
      hideBallTimeout = null;
    }
    showPopupNearBall();
  });

  // 鼠标移出小球后，延迟隐藏（避免闪烁）
  qaBall.addEventListener("mouseleave", () => {
    hideBallTimeout = setTimeout(() => {
      hidePopup();
      hideBall();
    }, 400);
  });

  document.body.appendChild(qaBall);
  return qaBall;
}

// -------------------- 3. 创建弹框 DOM --------------------
function createPopup() {
  if (qaPopup) return qaPopup;
  qaPopup = document.createElement("div");
  qaPopup.className = "qa-popup";
  qaPopup.style.display = "none";

  // 关闭按钮
  const close = document.createElement("div");
  close.className = "qa-popup-close";
  close.textContent = "×";
  close.addEventListener("click", () => {
    hidePopup();
    hideBall();
  });
  qaPopup.appendChild(close);

  // 内容容器
  const content = document.createElement("div");
  content.className = "qa-popup-content";
  qaPopup.appendChild(content);

  qaPopup.addEventListener("mouseenter", () => {
    if (hideBallTimeout) {
      clearTimeout(hideBallTimeout);
      hideBallTimeout = null;
    }
  });

  qaPopup.addEventListener("mouseleave", () => {
    hideBallTimeout = setTimeout(() => {
      hidePopup();
      hideBall();
    }, 400);
  });

  document.body.appendChild(qaPopup);
  return qaPopup;
}

// -------------------- 4. 根据划词位置显示小球 --------------------
function showBallAtSelection() {
  const selection = window.getSelection();
  if (!selection || selection.isCollapsed) {
    hideBall();
    hidePopup();
    return;
  }

  const text = selection.toString().trim();
  if (!text) {
    hideBall();
    hidePopup();
    return;
  }

  const range = selection.getRangeAt(0);
  const rect = range.getBoundingClientRect();
  if (
    !rect ||
    (rect.x === 0 && rect.y === 0 && rect.width === 0 && rect.height === 0)
  ) {
    hideBall();
    hidePopup();
    return;
  }

  const ball = createBall();
  // 小球出现在划词右上角偏上
  const offsetX = 8;
  const offsetY = -20;

  ball.style.left = window.scrollX + rect.right + offsetX + "px";
  ball.style.top = window.scrollY + rect.top + offsetY + "px";
  ball.dataset.selectionText = text; // 把选中的文本存下来
  ball.style.display = "flex";
}

// -------------------- 5. 隐藏小球 & 弹框 --------------------
function hideBall() {
  if (qaBall) {
    qaBall.style.display = "none";
  }
}

function hidePopup() {
  if (qaPopup) {
    qaPopup.style.display = "none";
  }
}

// -------------------- 6. 在小球附近展示弹框并填充内容 --------------------
function showPopupNearBall() {
  const ball = createBall();
  const popup = createPopup();
  const contentDiv = popup.querySelector(".qa-popup-content");

  // 先判断题库有没有加载到
  if (!QA_DATA || QA_DATA.length === 0) {
    contentDiv.innerHTML =
      '<div class="qa-popup-empty">题库未加载或为空（当前 0 条题）。<br>请确认 questions.json 放在扩展根目录，并在 manifest.json 的 web_accessible_resources 中声明。</div>';
    const ballRect = ball.getBoundingClientRect();
    popup.style.left = window.scrollX + ballRect.right + 8 + "px";
    popup.style.top = window.scrollY + ballRect.top + "px";
    popup.style.display = "block";
    return;
  }

  const selectedText = (ball.dataset.selectionText || "").trim();
  if (!selectedText) {
    contentDiv.innerHTML =
      '<div class="qa-popup-empty">没有检测到选中文本</div>';
  } else {
    // 查题
    const result = searchQuestion(selectedText);
    if (!result) {
      contentDiv.innerHTML =
        '<div class="qa-popup-empty">题库中未找到匹配题目<br>可以尝试：只划题干，不要把选项一起划上。</div>';
    } else {
      const { question, options, answer } = result;
      const optionsHtml = Object.entries(options || {})
        .map(([key, val]) => `<div>${key}. ${escapeHtml(val)}</div>`)
        .join("");

      contentDiv.innerHTML = `
        <div class="qa-popup-title">${escapeHtml(question)}</div>
        <div class="qa-popup-options">${optionsHtml}</div>
        <div class="qa-popup-answer">正确答案：${escapeHtml(answer)}</div>
      `;
    }
  }

  // 放在小球右侧
  const ballRect = ball.getBoundingClientRect();
  popup.style.left = window.scrollX + ballRect.right + 8 + "px";
  popup.style.top = window.scrollY + ballRect.top + "px";
  popup.style.display = "block";
}

// -------------------- 7. 题目检索逻辑（加强版） --------------------

// 归一化文本：去空格、去常见标点、去“以下/下列”这种废话
function normalizeText(str) {
  if (!str) return "";
  return String(str)
    .replace(/\s+/g, "")
    .replace(/[（）()【】\[\]。,，.？?！!；;：:、]/g, "")
    .replace(/以下[^\u4e00-\u9fa5]*/g, "")
    .replace(/下列[^\u4e00-\u9fa5]*/g, "");
}

function searchQuestion(selectedText) {
  if (!QA_DATA || QA_DATA.length === 0) return null;

  const rawSel = selectedText.trim();
  // 别把整页都算进去，只留前一段
  const cutSel = rawSel.slice(0, 60);
  const normSel = normalizeText(cutSel);
  if (!normSel) return null;

  // 1）优先：题干包含划词（或划词包含题干前面几字）
  let directMatch = QA_DATA.find((item) => {
    if (!item || !item.question) return false;
    const qNorm = normalizeText(item.question);
    if (!qNorm) return false;
    if (qNorm.includes(normSel)) return true;
    if (normSel.includes(qNorm.slice(0, Math.min(qNorm.length, 8)))) return true;
    return false;
  });

  if (directMatch) return directMatch;

  // 2）模糊匹配：统计 normSel 中有多少字符出现在题干里，取得分最高的
  let best = null;
  let bestScore = 0;

  QA_DATA.forEach((item) => {
    if (!item || !item.question) return;
    const qNorm = normalizeText(item.question);
    if (!qNorm) return;

    let score = 0;
    for (let ch of normSel) {
      if (qNorm.includes(ch)) score++;
    }

    if (score > bestScore) {
      bestScore = score;
      best = item;
    }
  });

  // 阈值随便给个 4，防止乱匹配（你以后可以自己调）
  if (best && bestScore >= 4) {
    return best;
  }

  return null;
}

// -------------------- 8. 工具函数：转义 HTML --------------------
function escapeHtml(str) {
  if (str == null) return "";
  return String(str)
    .replace(/&/g, "&amp;")
    .replace(/</g, "&lt;")
    .replace(/>/g, "&gt;")
    .replace(/"/g, "&quot;");
}

// -------------------- 9. 监听划词 --------------------
document.addEventListener("mouseup", () => {
  // 延迟一下，让 selection 更新
  setTimeout(showBallAtSelection, 10);
});

// 初始化加载题库
loadQuestions();
